package com.vmware.samples.customprovider;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.vmware.vise.data.PropertySpec;
import com.vmware.vise.data.query.DataException;
import com.vmware.vise.data.query.DataServiceExtensionRegistry;
import com.vmware.vise.data.query.PropertyProviderAdapter;
import com.vmware.vise.data.query.PropertyRequestSpec;
import com.vmware.vise.data.query.PropertyValue;
import com.vmware.vise.data.query.ResultItem;
import com.vmware.vise.data.query.ResultSet;
import com.vmware.vise.data.query.TypeInfo;

/**
 * Provider of the elevationData property for a HostSystem.
 * This example demonstrates the use of an external web service.
 */
public class CustomHostPropertyProviderImpl implements PropertyProviderAdapter {

   private static final Log _logger = LogFactory.getLog(CustomHostPropertyProviderImpl.class);

   /* vSphere type for a host */
   private static final String HOST_TYPE = "HostSystem";

   /*
    * Custom host property provided by this adapter. It must be qualified with its
    * own namespace to avoid collisions with the vSphere default space
    */
   private static final String ELEVATIONDATA_PROPERTY = "samples:elevationData";

   private GoogleService _googleService;

   /**
    *  Constructor used to inject the utility services (see the declaration
    *  in main/resources/spring/bundle-context-osgi.xml)
    *
    * @param googleService
    *    Web service example using the Google Elevation API.
    * @param registry
    *    Registration interface for data adapters and providers.
    */
   public CustomHostPropertyProviderImpl(GoogleService googleService,
         DataServiceExtensionRegistry registry) {
      _googleService = googleService;
      registry.registerDataAdapter(this, getProvidedTypeInfos());
   }

   /**
    * @return the types and properties handled by this adapter.
    */
   private TypeInfo[] getProvidedTypeInfos() {
      TypeInfo hostTypeInfo = new TypeInfo();
      hostTypeInfo.type = HOST_TYPE;
      hostTypeInfo.properties = new String[] { ELEVATIONDATA_PROPERTY };

      return new TypeInfo[] { hostTypeInfo };
   }

   /* (non-Javadoc)
    * @see com.vmware.vise.data.query.PropertyProviderAdapter#getProperties
    * (com.vmware.vise.data.query.PropertyRequestSpec)
    */
   @Override
   public ResultSet getProperties(PropertyRequestSpec propertyRequest) {
      // This is the main entry point of the PropertyProviderAdapter.
      // propertyRequest.properties contains the properties to be retrieved,
      // just "elevationData" in this example, and the EarthLocation parameter.

      if ((propertyRequest == null) ||
            (propertyRequest.objects == null) ||
            (propertyRequest.objects.length != 1)) {
         // This example only handles queries for 1 object at a time. To support
         // multiple objects it needs to handle multiple location parameters.
         _logger.error("CustomHostPropertyProvider only supports 1 object at a time.");
         return null;
      }
      Object hostRef = propertyRequest.objects[0];

      EarthLocation location = getLocationParameter(propertyRequest);
      if (location == null) {
         _logger.error("Invalid PropertyRequestSpec in CustomHostPropertyProvider");
         return null;
      }

      ResultSet result = new ResultSet();

      try {
         ElevationData data = _googleService.getElevationData(
               location.latitude, location.longitude);

         // Construct the ResultItem and ResultSet
         ResultItem ri = new ResultItem();
         ri.resourceObject = hostRef;
         PropertyValue pv = new PropertyValue();
         pv.resourceObject = hostRef;
         pv.propertyName = ELEVATIONDATA_PROPERTY;
         pv.value = data;
         ri.properties = new PropertyValue[] { pv };
         result.items = new ResultItem[] { ri };
      } catch (Exception e) {
         _logger.error("CustomHostPropertyProviderImpl.getProperties error", e);
         // Passing the exception in the result allows to display an error notification
         // in the client UI.
         DataException de = DataException.newInstance(e, propertyRequest.objects, 
               propertyRequest.properties[0].propertyNames);
         result.error = de;
      }
      return result;
   }

   /**
    * Extract the EarthLocation parameter from the PropertyRequestSpec
    * @param propertyRequest
    * @return the EarthLocation of the host
    */
   private EarthLocation getLocationParameter(PropertyRequestSpec propertyRequest) {
      if ((propertyRequest == null) || (propertyRequest.properties == null)
            || (propertyRequest.properties.length != 1)) {
         return null;
      }
      PropertySpec propertySpec = propertyRequest.properties[0];
      if ((propertySpec.propertyNames == null) ||
            (propertySpec.propertyNames.length != 1) ||
            (!ELEVATIONDATA_PROPERTY.equals(propertySpec.propertyNames[0]))) {
         return null;
      }
      if ((propertySpec.parameters == null) ||
            (propertySpec.parameters.length != 1)) {
         return null;
      }
      EarthLocation location = (EarthLocation)propertySpec.parameters[0].parameter;
      return location;
   }
}